#!/usr/local/bin/python3
# coding: utf8

import requests
import json
import const
import datetime


def change_ticket_status(ticket):
    """
    Mark ticket status as "in_progress" and comments than this alert has been forwarded to the hive
    """
    url = f"{const.API_ACTION_PLAN_ENDPOINT}/{ticket['_id']}"
    authentication_headers = dict(Authorization=f"Bearer {const.API_KEY}")

    ticket["status"] == "in_progress"
    journal = ticket.get("journals",[])
    journal.append({"notes":"Alert forwarded to the hive", "revision":0})
    r = requests.patch(url, headers={**{"If-Match": ticket["_etag"]}, **authentication_headers}, json={"status": "in_progress", "journals": journal})
    if r.status_code != 200 :
        print("Error while editing alert")
        return False
    else:
        print("Alert edited")
        return True

def get_all_serenety_scopes():
    """
    List all serenety scope for a user.
    """
    authentication_headers = dict(Authorization=f"Bearer {const.API_KEY}")
    params = {
        "page":1,
        "where": json.dumps({
            "type": "serenety"
        })
    }
    list_scopes_id = []
    # this variable will be used to count the number of result from the last query
    nb_items_last_occurence = 1
    while nb_items_last_occurence > 0 :

        r = requests.get(const.API_SERENETY_SCOPE_ENDPOINT, params=params, headers=authentication_headers)
        response_json = json.loads(r.text)
        scopes_jsons = response_json['_items']

        # if we have more than 25 perimeters
        for scope in scopes_jsons:
            list_scopes_id.append(scope["_id"])

        # we iterate over page by increaasing page number by 1
        params["page"] += 1
        # we stop when last request did not response anything
        nb_items_last_occurence = len(response_json['_items'])
    return list_scopes_id

def get_all_new_tickets():
    """
    List all ticket marked as new
    """
    authentication_headers = dict(Authorization=f"Bearer {const.API_KEY}")
    url = const.API_ACTION_PLAN_LIST_ENDPOINT
    list_scope_id = get_all_serenety_scopes()
    # We want to get only alert created in the last n hour
    now = datetime.datetime.now() - datetime.timedelta(hours=240)
    formated_from_date = now.strftime("%a, %d %b %Y %H:%M:%S GMT")

    params = {
        "page" : 1,
        "scope_id": ",".join(list_scope_id),
        "queryExtras" : json.dumps({"status":{"$eq":"new"}})
    }

     # this variable will be used to count the number of result from the last query
    nb_items_last_occurence = 1
    # we iterate over result and yields ticket one by one
    while nb_items_last_occurence > 0 :
        r = requests.get(url, headers=authentication_headers, params=params)
        response_json = json.loads(r.text)
        tickets = response_json['_items']
        for ticket in tickets:
            yield ticket
        nb_items_last_occurence = len(response_json['_items'])
        params["page"] += 1
    return

def get_severity_ticket(ticket):
    """
    Translate XMCO severity to a integer
    """
    XMCO_SEVERITY_TO_HIVE_SEVERITY = {
        "low": 1,
        "medium": 2,
        "high": 3,
        "critical" : 4
    }
    return XMCO_SEVERITY_TO_HIVE_SEVERITY.get(ticket.get("severity", "low"))

def push_ticket_to_the_hive(ticket):
    """
    Forward a ticket to the hive
    """

    THE_HIVE_HEADERS = {"Authorization" : f"Bearer {const.AUTH_HIVE_TOKEN}"}
    incident = {
        "title": ticket["title"],
        "description": ticket["description"],
        "severity": get_severity_ticket(ticket),
        # We could also use field like custom_fields.category to set more tag
        "tags": [
          "serenety"
        ],
        "status": "New",
        # We could create custom field to store alerte category, subcategory and observable
        # Custom fields need to be created via gui before
        # see. https://docs.thehive-project.org/thehive/user-guides/administrators/custom-fields/
        "customFields": {
            "category": ticket["custom_fields"]["category"],
            "subcategory": ticket["custom_fields"]["subcategory"]
        }
        # List of impacted assets is present in the field ticket["assets"]
      }
    return requests.post(const.THE_HIVE_CREATE_ENDPOINT,json=incident, headers=THE_HIVE_HEADERS)

if __name__ == '__main__':
    # Put all new tickets in the hive
    # Update status and had a comment
    for ticket in get_all_new_tickets():
        change_ticket_status(ticket)
        print(push_ticket_to_the_hive(ticket))
