#!/usr/local/bin/python3
# coding: utf8

import requests
import json
from prettytable import PrettyTable
import const


def get_watch_advisories_by_scope(scope_id):
    """
    Get the 25 most recent security watch advisories associated to a specific scope:

    To do that, we first need to list the CPE associated with our scope in what we call a
    fake asset. This asset contains all CPE used in the scope, meaning all the technologies followed.

    Then, we will filter the XMCO database of advisories with the CPE names extracted.

    author: ebaudin
    """

    print("--- Get Watch Advisories By Scope ---")

    # API key could be obtained as described in the documentation : https://leportail.xmco.fr/doc/api
    # We recommend using a dedicated service account when used in script
    # Edit const to use your own API key
    authentication_headers = dict(Authorization=f"Bearer {const.API_KEY}")

    params = {
        "where": json.dumps({
            "scopes": scope_id,
            "asset_type": "fake"
        })
    }
    r = requests.get(const.API_ASSET_ENDPOINT, params=params, headers=authentication_headers)

    cpe_names = json.loads(r.text)['_items'][0]["cpe_names"]
    if not cpe_names:
        print("There is no technologies in this scope.")
        return
    
    params = {
        "where": json.dumps({
            "cpe_names": {
                "$in": cpe_names
            }
        }),
        "sort": "-_created"
    }
    r = requests.get(const.API_ADVISORY_ENDPOINT, params=params, headers=authentication_headers)

    advisories_json = json.loads(r.text)['_items']

    print("\n[+] Preview of advisories associated with the scope %s:" % scope_id)
    table = PrettyTable(['Type', 'Severity', 'Title'])
    table.align = 'l'

    for advisory in advisories_json:
        table.add_row([advisory["advisory_type"], advisory["severity"].upper(), advisory["content_fr"]["title"]])

    print(table)


if __name__ == '__main__':
    # The following value is an XMCO scope for presentation purposes,
    # Change this value with one of your scope `_id`.

    xmco_scope = "5d6e7833d06a8f000c21a12d"
    get_watch_advisories_by_scope(xmco_scope)
